package uk.ac.ox.cs.krr.chebiclassifier;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import org.openscience.cdk.ChemFile;
import org.openscience.cdk.exception.CDKException;
import org.openscience.cdk.interfaces.IAtom;
import org.openscience.cdk.interfaces.IAtomContainer;
import org.openscience.cdk.interfaces.IBond;
import org.openscience.cdk.interfaces.IChemModel;
import org.openscience.cdk.io.MDLV2000Reader;
import org.openscience.cdk.tools.manipulator.AtomContainerManipulator;

public class DGGenerator {
	
	//given a molfile it produces an object of the
	//description graph class
	public DescriptionGraph buildDescriptionGraph(File molfile){
		String molfileContent=retrieveContent(molfile);
		Map<Integer,String> nodes=new HashMap<Integer,String>();
		Set<Edge> edges=new  LinkedHashSet<Edge>();
		String moleculeName="";
		try{
			moleculeName=retrieveChEBIID(molfile);
			//convert into lower case to fit XSB syntax
			moleculeName=moleculeName.toLowerCase();
			MDLV2000Reader reader = new MDLV2000Reader(new StringReader(molfileContent)) ;
			ChemFile chemFile = (ChemFile) reader.read(new ChemFile());
			IChemModel model = chemFile.getChemSequence(0).getChemModel(0);
			IAtomContainer molContent = model.getMoleculeSet().getAtomContainer(0);
			AtomContainerManipulator.percieveAtomTypesAndConfigureAtoms(molContent);
			nodes=retrieveNodes(molContent,moleculeName);
			edges=retrieveEdges(molContent,moleculeName);
		} catch (NullPointerException npe) {
			npe.printStackTrace();
		} catch (CDKException e) {
			e.printStackTrace();
		}	
		return new DescriptionGraph(nodes,edges,moleculeName);
	}
	
	//it returns the content of a molfile in the form of a string
	public String retrieveContent(File molfile){
		StringBuffer content=new StringBuffer();
        String CRLF=System.getProperty("line.separator");
        try {
			BufferedReader input=new BufferedReader(new FileReader(molfile)) ;
			String line="";
			while(!(line.endsWith("END"))){
				line=input.readLine();
				content.append(line);
				if (!(line.endsWith("END"))){
					content.append(CRLF);
				}
			}
		}
        catch (FileNotFoundException e) {
			e.printStackTrace();
		} 
        catch (IOException e) {
			e.printStackTrace();
		}
		return content.toString();
	}
	
	//it returns a map with labeled nodes
	public Map<Integer,String> retrieveNodes(IAtomContainer molContent,String moleculeName){
		Map<Integer,String> nodes=new HashMap<Integer,String>();
		try{
			nodes.put(new Integer(0),moleculeName);
			int numberOfAtoms=molContent.getAtomCount();
			for (int i=0; i<numberOfAtoms; i++){
				//System.out.println("Atom id: "+molContent.getAtom(i).getListenerCount());
				int nodeIndex=i+1;
				nodes.put(new Integer(nodeIndex),molContent.getAtom(i).getSymbol().toLowerCase());
			}
		}
		catch(Exception e){
			e.printStackTrace();
		}		
		return nodes;		
	}
	
	//it returns a set of labeled edges
	public Set<Edge> retrieveEdges(IAtomContainer molContent,String moleculeName){
		Set<Edge> edges=new LinkedHashSet<Edge>();
		try {
			//keep the indices of the atoms from molContent
			Map<IAtom,Integer> atomIndexMap = new HashMap<IAtom, Integer>();
			int numberOfAtoms=molContent.getAtomCount();
			for (int i=0; i<numberOfAtoms; i++){
				atomIndexMap.put(molContent.getAtom(i), new Integer(i+1));
			}
			Integer rootNode=new Integer(0);
			//first add the hasAtom edges
			for (int i=0; i<numberOfAtoms; i++){
				int nodeIndex=i+1;
				Edge edge=new Edge(rootNode,new Integer(nodeIndex),"hasAtom");
				edges.add(edge);
			}
			//then add the bond edges
			int numberOfBonds=molContent.getBondCount();
			for (int i=0; i<numberOfBonds; i++){
				IBond bond=molContent.getBond(i);
				Integer fromNode=atomIndexMap.get(bond.getAtom(0));
				Integer toNode=atomIndexMap.get(bond.getAtom(1));
				String bondOrder=bond.getOrder().toString().toLowerCase();
				Edge edgeForward=new Edge(fromNode,toNode,bondOrder);
				Edge edgeBackward=new Edge(toNode,fromNode,bondOrder);
				edges.add(edgeForward);
				edges.add(edgeBackward);
			}
		}
		catch(Exception e){
			e.printStackTrace();
		}	
		return edges;
		
	}
	
	//it returns the chebi ID, e.g. ChEBI_12345
	public String retrieveChEBIID(File molfile){
		return molfile.getName().replaceFirst("\\.mol", "");
	}
	
}
