package uk.ac.ox.cs.krr.dlvstructured;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.Set;

/* This class generates (sets of) objects of the java class 
 * DLVRule for the representation of description graphs.
*/

public class DLVRulesGenerator {
	
	protected enum TermMode { FUNCTION_TERMS_ON, VARIABLES_ON, CONSTANTS_ON }
	
	//given a description graph it returns its translation into LP rules	
	public Set<DLVRule>  translateDGintoRules(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		rules.addAll(produceLayoutRules(descriptionGraph,false));
		rules.add(produceStartConceptFact(descriptionGraph.getStartConcept()));
		return rules;
	}
	
	//given a description graph it returns its translation into datalog rules for MSA check
	public Set<DLVRule>  translateDGintoMSARules(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		
		//rules.add(produceMSAStartRule(descriptionGraph));
		rules.addAll(produceLayoutRules(descriptionGraph,true));
		rules.addAll(produceSuccessionRules(descriptionGraph));
		rules.addAll(produceTagRules(descriptionGraph));
		rules.addAll(produceCycleDetectionRules(descriptionGraph));
		rules.add(produceStartConceptFact(descriptionGraph.getStartConcept()));
		
		return rules;
	}
	
	//given a 'dual' description graph, i.e. that represents a functional group,
	//it returns its translation into LP rules	
	public Set<DLVRule>  translateDualDGintoRules(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		rules.addAll(produceLayoutRulesForDualDG(descriptionGraph,false));
		rules.addAll(produceRecognitionRulesForDualDG(descriptionGraph));
		rules.add(produceStartConceptFact(descriptionGraph.getStartConcept()));
		return rules;
	}
	
	//given a 'dual' description graph, i.e. that represents a functional group,
	//it returns its translation into concept fact and recognition rules	
	public Set<DLVRule>  translateDualDGintoRecRulesFact(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		rules.addAll(produceRecognitionRulesForDualDG(descriptionGraph));
		rules.add(produceStartConceptFact(descriptionGraph.getStartConcept()));
		return rules;
	}
		
	//given a 'dual' description graph, i.e. that represents a functional group,
	//it returns its translation into generation rules	
	public Set<DLVRule>  translateDualDGintoGenRules(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		rules.addAll(produceLayoutRulesForDualDG(descriptionGraph,false));
		return rules;
	}
	
	//given a 'dual' description graph, i.e. that represents a functional group,
	//it returns its translation into datalog rules for MSA check	
	public Set<DLVRule>  translateDualDGintoMSARules(DescriptionGraph descriptionGraph){
			Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
			rules.addAll(produceLayoutRulesForDualDG(descriptionGraph,true));
			rules.addAll(produceSuccessionRulesForDualDG(descriptionGraph));
			rules.addAll(produceTagRulesForDualDG(descriptionGraph));
			rules.addAll(produceCycleDetectionRules(descriptionGraph));
			rules.addAll(produceRecognitionRulesForDualDG(descriptionGraph));
			rules.add(produceStartConceptFact(descriptionGraph.getStartConcept()));
			return rules;
	}
	
	//given a description graph it returns its translation into LP rules without
	//the start concept fact
	public Set<DLVRule>  translateDGintoRulesNoFact(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		rules.addAll(produceLayoutRules(descriptionGraph,false));
		return rules;
	}
	
public Set<DLVRule> produceLayoutRules(DescriptionGraph descriptionGraph,boolean forMSACheck){
		
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
	    
		bodyAtoms.add(produceStartAtom(descriptionGraph.getStartConcept()));
	    
		//create all rules that give types to the nodes of the graph
	    for (Node node:descriptionGraph.getNodes()){
	    	//System.out.println("Producing layout rule for node "+node.getNumber().toString());
	    	Set<String> label=node.getLabel();
	    	//System.out.println("Size of the label is "+label.size());
	    	for (String unaryPredicate:label){
	    		ArrayList<String> headAtomTerms=new ArrayList<String>();
	    		String term;
	    		if (!forMSACheck){
	    			term=buildFunctionLayoutTerm(descriptionGraph.getStartConcept(), node.getNumber());
	    		}
	    		else {
	    			term=buildConstantLayoutTerm(descriptionGraph.getStartConcept(), node.getNumber());
	    		}
	    		headAtomTerms.add(term);
	       		Atom headAtom=new Atom(unaryPredicate,headAtomTerms);
	    		//System.out.println(new DLVRule(headAtom,bodyAtoms).getDLVSyntaxRule());
	    		rules.add(new DLVRule(headAtom,bodyAtoms));
	    	}
	    }	    
	    
	    //create all rules that give types to the edges of the graph
	    for (Edge edge:descriptionGraph.getEdges()){
	    	//System.out.println("Producing layout rule for edge "+edge.getFromNode().toString()
	    	//		+"->"+edge.getToNode().toString());
	    	Set<String> label=edge.getLabel();
	    	//System.out.println("Size of the label is "+label.size());
	    	for (String binaryPredicate:label){
	    		ArrayList<String> headAtomTerms=new ArrayList<String>();
	    		String fromTerm,toTerm;
	    		if (!forMSACheck){
	    			fromTerm=buildFunctionLayoutTerm(descriptionGraph.getStartConcept(), edge.getFromNode());
	    			toTerm=buildFunctionLayoutTerm(descriptionGraph.getStartConcept(), edge.getToNode());
	    		}
	    		else {
	    			fromTerm=buildConstantLayoutTerm(descriptionGraph.getStartConcept(), edge.getFromNode());
	    			toTerm=buildConstantLayoutTerm(descriptionGraph.getStartConcept(), edge.getToNode());
	    		}
	    		headAtomTerms.add(fromTerm);
	    		headAtomTerms.add(toTerm);
	    		Atom headAtom=new Atom(binaryPredicate,headAtomTerms);
	    		//System.out.println(new DLVRule(headAtom,bodyAtoms).getDLVSyntaxRule());
	    		rules.add(new DLVRule(headAtom,bodyAtoms));
	    	}
	    }	    
	   return rules;
	}

	//it produces a set of rules of the form 
	//alcohol(X), not r_alcohol(X) -> hasAtom(X,f_alcohol_1(X))
	//for the labels of all nodes and all edges of the description graph
	//and of the form alcohol(X), not r_alcohol(X) -> g_alcohol(f_alcohol_1(X))
	//for all generated nodes of the description graph
	public Set<DLVRule> produceLayoutRulesForDualDG(DescriptionGraph descriptionGraph,boolean forMSACheck){
		String startConcept=descriptionGraph.getStartConcept();
		
		//first retrieve all layout rules as produced for a plain description graph
		Set<DLVRule> rules=produceLayoutRules(descriptionGraph,forMSACheck);
		Set<DLVRule> newRules=new LinkedHashSet<DLVRule>();
		ArrayList<Atom> negativeBodyAtoms=new ArrayList<Atom>();
		
		//then add new rules that tag the generated functional terms
		int numberOfNodes=descriptionGraph.getNodes().size();
		for (int i=1; i<=numberOfNodes; i++){
			ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
			bodyAtoms.add(produceStartAtom(startConcept));
			if (forMSACheck)
				rules.add(new DLVRule(produceGenerationAtom(startConcept, i,TermMode.CONSTANTS_ON), bodyAtoms));
			else
				rules.add(new DLVRule(produceGenerationAtom(startConcept, i,TermMode.FUNCTION_TERMS_ON), bodyAtoms));			
		}
		
		//then enrich the bodies of all the above rules with negative atoms
		negativeBodyAtoms.add(produceRecognitionAtom(startConcept));
		
		for (DLVRule rule:rules){
			DLVRule newRule=new DLVRule(rule.getHeadAtom(), rule.getBodyAtoms(), negativeBodyAtoms);
			newRules.add(newRule);	
			//System.out.println("extra printing+++++++++++++++++");
			//System.out.println(newRule.getDLVSyntaxRule());
		}	
		
		return newRules;
	}
	
	//it produces a set consisting of two rules 
	//one of the form 
	//hasAtom(X,Y1),..., not g_alcohol(Y1),... -> alcohol(X)
	//and one of the form
	//hasAtom(X,Y1),..., not g_alcohol(Y1),... -> r_alcohol(X)
	//where the body results from the labels of all nodes and all edges of the description graph
	public Set<DLVRule> produceRecognitionRulesForDualDG(DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
		String startConcept=descriptionGraph.getStartConcept();
		ArrayList<Atom> negativeBodyAtoms=new ArrayList<Atom>();
		int numberOfNodes=descriptionGraph.getNodes().size();
		
		//add to the set of negative body atoms all generation atoms
		for (int i=1; i<=numberOfNodes; i++)
			negativeBodyAtoms.add(produceGenerationAtom(startConcept,i,TermMode.VARIABLES_ON));
		
		//create all atoms that give types to the nodes of the graph
	    for (Node node:descriptionGraph.getNodes()){
	    	Set<String> label=node.getLabel();
	    	for (String unaryPredicate:label){
	    		ArrayList<String> terms=new ArrayList<String>();
	    		String term=buildVariable(node.getNumber());
	    		terms.add(term);
	       		bodyAtoms.add(new Atom(unaryPredicate,terms));
	    	}
	    }	    
	    
	    //create all atoms that give types to the edges of the graph
	    for (Edge edge:descriptionGraph.getEdges()){
	    	Set<String> label=edge.getLabel();
	    	for (String binaryPredicate:label){
	    		ArrayList<String> terms=new ArrayList<String>();
	    		String fromTerm=buildVariable(edge.getFromNode());
	    		String toTerm=buildVariable(edge.getToNode());
	    		terms.add(fromTerm);
	    		terms.add(toTerm);
	    		bodyAtoms.add(new Atom(binaryPredicate,terms));
	    	}
	    }
	    rules.add(new DLVRule(produceStartAtom(startConcept),bodyAtoms,negativeBodyAtoms));
	    rules.add(new DLVRule(produceRecognitionAtom(startConcept),bodyAtoms,negativeBodyAtoms));
	    
	    //for (DLVRule r:rules)
	    //	System.out.println(r.getDLVSyntaxRule());
	    
	    return rules;
	}
	
	//produces an atom of the form r_alcohol(X)
	public Atom produceRecognitionAtom(String startConcept){
		//create the predicate
		StringBuffer sb=new StringBuffer();
		sb.append("r_");
		sb.append(startConcept);
		String predicate=sb.toString();
		clearBuffer(sb);
		//create the terms
		ArrayList<String> terms=new ArrayList<String>();
		terms.add("X");
	    return new Atom(predicate,terms);
	}
	
	//produces an atom of the form g_alcohol(f_alcohol_1(X)) if function is on
	//otherwise of the form g_alcohol(Y1) if function is off
	public Atom produceGenerationAtom(String startConcept,int nodeNumber,TermMode termMode){
		//create the predicate
		StringBuffer sb=new StringBuffer();
		sb.append("g_");
		sb.append(startConcept);
		String predicate=sb.toString();
		clearBuffer(sb);
		//create the terms
		ArrayList<String> terms=new ArrayList<String>();
		if (TermMode.FUNCTION_TERMS_ON.equals(termMode))
			terms.add(buildFunctionLayoutTerm(startConcept, nodeNumber));
		else if (TermMode.VARIABLES_ON.equals(termMode))
			terms.add(buildVariable(nodeNumber));
		else if (TermMode.CONSTANTS_ON.equals(termMode))
			terms.add(buildConstantLayoutTerm(startConcept, nodeNumber));
		return new Atom(predicate,terms);
	}

	
	//it produces a term of the form f_startConcept_1(X)
	//unless nodeNumber is 0 in which case it produces X
	public String buildFunctionLayoutTerm(String startConcept,int nodeNumber){
		StringBuffer sb=new StringBuffer();
		if (nodeNumber!=0){
			sb.append("f_");
			sb.append(startConcept);
			sb.append("_");
			sb.append(nodeNumber);
			sb.append("(X)");
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}
		else {
			sb.append("X");
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}			
	}

	//it produces a term of the form c_startConcept_1
	//unless nodeNumber is 0 in which case it produces X
	public String buildConstantLayoutTerm(String startConcept,int nodeNumber){
		StringBuffer sb=new StringBuffer();
		if (nodeNumber!=0){
			sb.append("c_");
			sb.append(startConcept);
			sb.append("_");
			sb.append(nodeNumber);
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}
		else {
			sb.append("X");
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}			
	}
	
	//it produces a variable of the form Y1
	//unless nodeNumber is 0 in which case it produces X
	public String buildVariable(int nodeNumber){
		StringBuffer sb=new StringBuffer();
		if (nodeNumber!=0){
			sb.append("Y");
			sb.append(nodeNumber);
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}
		else {
			sb.append("X");
			String term=sb.toString();
			clearBuffer(sb);
			return term;
		}			
	}	
	
	//it produces an atom of the form startConcept(X)
	public Atom produceStartAtom(String startConcept){
		ArrayList<String> bodyAtomTerms=new ArrayList<String>();
	    bodyAtomTerms.add("X");    
	    return new Atom(startConcept,bodyAtomTerms);
	    
	}
	
	//it produces a fact of the form startConcept(c_startConcept)	
	public DLVRule produceStartConceptFact(String startConcept){
		StringBuffer sb=new StringBuffer();
		ArrayList<String> headAtomTerms=new ArrayList<String>();
		sb.append("c_");
		sb.append(startConcept);
		headAtomTerms.add(sb.toString());
		Atom headAtom=new Atom(startConcept,headAtomTerms);
	    ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
		return new DLVRule(headAtom,bodyAtoms);
	}
	
	
	
	//given a description graph it returns all the succession rules of the form
	//chebi_12345(X) -> successor(X,c_chebi_12345_1)
	public Set<DLVRule>  produceSuccessionRules (DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		StringBuffer sb=new StringBuffer();
		
		ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
	    bodyAtoms.add(produceStartAtom(descriptionGraph.getStartConcept()));
		
		for (int i=1; i<descriptionGraph.getNodes().size(); i++){
	    	ArrayList<String> headAtomTerms=new ArrayList<String>();
	        sb.append("X");
	        headAtomTerms.add(sb.toString());
	        clearBuffer(sb);
	        headAtomTerms.add(buildConstantLayoutTerm(descriptionGraph.getStartConcept(), i));
	        Atom headAtom=new Atom("successor",headAtomTerms);
	    	rules.add(new DLVRule(headAtom,bodyAtoms));
	    	
	    }	
		
		return rules;
	}
	
	//given a description graph it returns all the tag rules of the form
	//chebi_12345(X) -> p_chebi_12345_1(c_chebi_12345_1)
	public Set<DLVRule>  produceTagRules (DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		StringBuffer sb=new StringBuffer();
		
		ArrayList<Atom> bodyAtoms=new ArrayList<Atom>();
		bodyAtoms.add(produceStartAtom(descriptionGraph.getStartConcept()));
		
		for (int i=1; i<descriptionGraph.getNodes().size(); i++){
	    	ArrayList<String> headAtomTerms=new ArrayList<String>();
	    	headAtomTerms.add(buildConstantLayoutTerm(descriptionGraph.getStartConcept(), i));
	        sb.append("p_");
	    	sb.append(descriptionGraph.getStartConcept());
	    	sb.append("_");
	    	sb.append(i);
	    	String tag=sb.toString();
	    	clearBuffer(sb);
	        Atom headAtom=new Atom(tag,headAtomTerms);
	        rules.add(new DLVRule(headAtom,bodyAtoms));	    	
	    }			
		return rules;
	}
	
	//given a description graph it returns all the succession rules of the form
	//alcohol(X),not r_alcohol(X) -> successor(X,c_alcohol_1)
	public Set<DLVRule>  produceSuccessionRulesForDualDG (DescriptionGraph descriptionGraph){
		String startConcept=descriptionGraph.getStartConcept();
		
		//first retrieve all succession rules as produced for a plain description graph
		Set<DLVRule> rules=produceSuccessionRules(descriptionGraph);
		Set<DLVRule> newRules=new LinkedHashSet<DLVRule>();
		
		//then enrich the bodies of all the above rules with negative atoms
		ArrayList<Atom> negativeBodyAtoms=new ArrayList<Atom>();
		negativeBodyAtoms.add(produceRecognitionAtom(startConcept));
				
		for (DLVRule rule:rules){
			DLVRule newRule=new DLVRule(rule.getHeadAtom(), rule.getBodyAtoms(), negativeBodyAtoms);
			newRules.add(newRule);			
		}	
		
		return newRules;
	}
		
	//given a description graph it returns all the tag rules of the form
	//alcohol(X),not r_alcohol(X) -> p_chebi_12345_1(c_chebi_12345_1)
	public Set<DLVRule>  produceTagRulesForDualDG (DescriptionGraph descriptionGraph){
		String startConcept=descriptionGraph.getStartConcept();
		
		//first retrieve all succession rules as produced for a plain description graph
		Set<DLVRule> rules=produceTagRules(descriptionGraph);
		Set<DLVRule> newRules=new LinkedHashSet<DLVRule>();
		
		//then enrich the bodies of all the above rules with negative atoms
		ArrayList<Atom> negativeBodyAtoms=new ArrayList<Atom>();
		negativeBodyAtoms.add(produceRecognitionAtom(startConcept));
				
		for (DLVRule rule:rules){
			DLVRule newRule=new DLVRule(rule.getHeadAtom(), rule.getBodyAtoms(), negativeBodyAtoms);
			newRules.add(newRule);			
		}	
		
		return newRules;
	}
	
	
	//given a description graph it returns all the cycle
	//detection rules of the form
	//p_chebi_12345_1(X),descendant(X,Y),p_chebi_12345_1(Y)-> cycle
	public Set<DLVRule>  produceCycleDetectionRules (DescriptionGraph descriptionGraph){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		ArrayList<String> cycleAtomTerms=new ArrayList<String>();
		Atom cycleAtom=new Atom("cycle",cycleAtomTerms);
		
		for (int i=1; i<descriptionGraph.getNodes().size(); i++){
    		ArrayList<Atom> bodyAtomsCycleRule=new ArrayList<Atom>();
    		String tag="p_"+descriptionGraph.getStartConcept()+"_"+i;
    		
    		ArrayList<String> bodyAtomLeftCycleRuleTerms=new ArrayList<String>();
    		bodyAtomLeftCycleRuleTerms.add("X");
    		Atom leftBodyAtom=new Atom(tag,bodyAtomLeftCycleRuleTerms);
    		bodyAtomsCycleRule.add(leftBodyAtom);
    		
    		ArrayList<String> bodyAtomMidCycleRuleTerms=new ArrayList<String>();
    		bodyAtomMidCycleRuleTerms.add("X");
    		bodyAtomMidCycleRuleTerms.add("Y");
    	    Atom headAtomPropRule=new Atom("descendant",bodyAtomMidCycleRuleTerms);    		
    		bodyAtomsCycleRule.add(headAtomPropRule);
    		
    		ArrayList<String> bodyAtomRightCycleRuleTerms=new ArrayList<String>();
    		bodyAtomRightCycleRuleTerms.add("Y");
    		Atom rightBodyAtom=new Atom(tag,bodyAtomRightCycleRuleTerms);
    		bodyAtomsCycleRule.add(rightBodyAtom);    
    		
    		rules.add(new DLVRule(cycleAtom,bodyAtomsCycleRule));
    	}			
		return rules;
	}
	
	//for a set of description graphs it returns for each description graph
	//rules of the form  p_chebi_12345_1(X),descendant(X,Y),p_chebi_12345_1(Y)-> cycle
	//and the propagation and transitive closure rules for cycle detection
	public Set<DLVRule>  produceDescendantTransClosureRules (){
		Set<DLVRule> rules=new LinkedHashSet<DLVRule>();
		
		ArrayList<Atom> bodyAtomsPropRule=new ArrayList<Atom>();
		ArrayList<String> bodyAtomPropRuleTerms=new ArrayList<String>();
		bodyAtomPropRuleTerms.add("X");
		bodyAtomPropRuleTerms.add("Y");
	    bodyAtomsPropRule.add(new Atom("successor",bodyAtomPropRuleTerms));
	    
	    ArrayList<String> headAtomPropRuleTerms=new ArrayList<String>();
		headAtomPropRuleTerms.add("X");
		headAtomPropRuleTerms.add("Y");
	    Atom headAtomPropRule=new Atom("descendant",headAtomPropRuleTerms);
	    
	    rules.add(new DLVRule(headAtomPropRule,bodyAtomsPropRule));
	    
	    ArrayList<Atom> bodyAtomsTransRule=new ArrayList<Atom>();
		ArrayList<String> bodyAtomLeftTransRuleTerms=new ArrayList<String>();
		bodyAtomLeftTransRuleTerms.add("X");
		bodyAtomLeftTransRuleTerms.add("Y");
	    bodyAtomsTransRule.add(new Atom("descendant",bodyAtomLeftTransRuleTerms));
	    ArrayList<String> bodyAtomRightTransRuleTerms=new ArrayList<String>();
		bodyAtomRightTransRuleTerms.add("Y");
		bodyAtomRightTransRuleTerms.add("Z");
	    bodyAtomsTransRule.add(new Atom("successor",bodyAtomRightTransRuleTerms));
	    
	    
	    ArrayList<String> headAtomTransRuleTerms=new ArrayList<String>();
		headAtomTransRuleTerms.add("X");
		headAtomTransRuleTerms.add("Z");
	    Atom headAtomTransRule=new Atom("descendant",headAtomTransRuleTerms);
	    
	    rules.add(new DLVRule(headAtomTransRule,bodyAtomsTransRule));
	    
	    return rules;
	}
	
	//this methods clears the content of a string buffer
	public StringBuffer clearBuffer(StringBuffer sb){
		return sb.delete(0,sb.length());
	}

}
