package uk.ac.ox.cs.krr.dlvstructured;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Set;

/*This class handles operations that involve computation of the stable model for a dlv program.*/

public class ModelComputer {
	
	private String m_dlvFilesPath;
	private String m_dlvEnginePath;
	private long m_modelComputationTime;
	private long m_msaCheckTime;
	
	public ModelComputer(String dlvFilesPath,String dlvEnginePath){
		this.m_dlvFilesPath=dlvFilesPath;
		this.m_dlvEnginePath=dlvEnginePath;
	}

	//this method returns true only if all the programs 
	//represented by dlvProgramsForMSACheckNames are MSA
	public boolean areMSA(Set<String> dlvProgramsForMSACheckNames){
		boolean areMSA=true;
		
		for (String program:dlvProgramsForMSACheckNames)
			areMSA=areMSA & isMSA(program);
				
		return areMSA;
	}
	
	//this method tells whether the logic program with the name 
	//dlvProgramForMSACheckName is model-summarising acyclic 
	public boolean isMSA(String dlvProgramForMSACheckName){
		
		long start=System.currentTimeMillis();
		StableModel stableModel=computeDLVModel(dlvProgramForMSACheckName);
		Atom cycle=new Atom("cycle",new ArrayList<String>());
		long end=System.currentTimeMillis();
		m_msaCheckTime=end-start;
		return !stableModel.isEntailed(cycle);
	}

	//this method returns the time taken for the most recent
	//model computation
	public double getModelComputationTimeInSeconds(){
		return ((double)m_modelComputationTime)/1000;
	}
	
	//this method returns the time taken for the most recent
	//msa check
	public double getMSACheckTimeInSeconds(){
		return ((double)m_msaCheckTime)/1000;
	}
	
	//this method computes the dlv models of the dlv programs
	//whose names are stored in dlvProgramsNames and returns 
	//an arraylist that contains the stable models in the same order
	public ArrayList<StableModel> computeDLVModels(ArrayList<String> dlvProgramsNames){
		ArrayList<StableModel> stableModels=new ArrayList<StableModel>();
		
		for (int i=0; i<dlvProgramsNames.size(); i++)
			stableModels.add(computeDLVModel(dlvProgramsNames.get(i)));
					
		return stableModels;
	}
	
	//this method computes in two steps the dlv models of the dlv programs 
	//whose names are stored in dlvProgramsNames and returns 
	//an arraylist that contains the stable models in the same order
	public ArrayList<StableModel> computeDLVModelsStrat(ArrayList<String> dlvProgramsNames,
			String dlvGenRulesClassesFileName){
		ArrayList<StableModel> stableModels=new ArrayList<StableModel>();
		
		for (int i=0; i<dlvProgramsNames.size(); i++)
			stableModels.add(computeDLVModelStrat(dlvProgramsNames.get(i),dlvGenRulesClassesFileName));
					
		return stableModels;
	}
	
	//this method invokes the DLV engine that computes the stable model
	//of the specified DLV program and returns an object of the class 
	//StableModel that is constructed using the generated stable model
	public StableModel computeDLVModel(String dlvProgramName){
		String outputString;
		String computedModelFileName=m_dlvFilesPath+"/"+dlvProgramName+"Model";
		String CRLF=System.getProperty("line.separator");
		
		try {
			String commandLine="./"+m_dlvEnginePath +" -silent "+dlvProgramName;
			String[] commandArguments={""};
			File workingDirectory = new File(m_dlvFilesPath+"/");
			long start=System.currentTimeMillis();
			Process process = Runtime.getRuntime().exec(commandLine,commandArguments,workingDirectory);
			
			BufferedReader bufferedReader= new BufferedReader(new InputStreamReader(process.getInputStream()));
			//System.out.println("Creating file "+computedModelFileName+"...");
			File model = new File(computedModelFileName);
			model.createNewFile();
			FileWriter fileWriter = new FileWriter(model);
			BufferedWriter modelBufferedWriter = new BufferedWriter(fileWriter);
			
			while ((outputString = bufferedReader.readLine()) != null) {
				modelBufferedWriter.append(outputString);
				modelBufferedWriter.append(CRLF);
			}		
			
			modelBufferedWriter.close();
			long end=System.currentTimeMillis();
			m_modelComputationTime=end-start;			
		} catch (Exception e) {
			e.printStackTrace();
		}
		return new StableModel(computedModelFileName,true);
	}
	
	//this method invokes the DLV engine that 
	//first computes the stable model of the lower stratum DLV program 
	//then converts the result into a facts format file valid for DLV input
	//then computes the stable model of the facts format file and the upper stratum DLV program 
	//finally returns a StableModel that is constructed using the generated stable model
	public StableModel computeDLVModelStrat(String dlvProgramStratFileName,String dlvGenRulesClassesFileName){
			String outputString;
			
			//set the computed model file name, i.e. if dlvProgramStratFileName is jMolecules50_50_strat
			//then computedModelFileName becomes jMolecules50_50Model
			int lastIndex=dlvProgramStratFileName.length()-6;
			String computedModelFileName=m_dlvFilesPath+"/"+dlvProgramStratFileName.substring(0,lastIndex)+"Model";
			String CRLF=System.getProperty("line.separator");
			
			try {
				long start=System.currentTimeMillis();	
				//compute the stable model of dlvProgramStratFileName
				StableModel stableModelStrat=computeDLVModel(dlvProgramStratFileName);
				//create the file resulting from the stable model of dlvProgramStratFileName
				//and valid for DLV input
				String factsFormatFileName=dlvProgramStratFileName+"Facts";
				stableModelStrat.createFactsFormatFile(m_dlvFilesPath+"/"+factsFormatFileName);
				//compute the stable model of the facts resulting from  
				//the dlvProgramStratFileName and the program resulting from
				//the dlvGenRulesClassesFileName
				String commandLine="./"+m_dlvEnginePath +" -silent "+factsFormatFileName+" "+dlvGenRulesClassesFileName;
				String[] commandArguments={""};
				File workingDirectory = new File(m_dlvFilesPath+"/");
				
				Process process = Runtime.getRuntime().exec(commandLine,commandArguments,workingDirectory);
				
				BufferedReader bufferedReader= new BufferedReader(new InputStreamReader(process.getInputStream()));
				//System.out.println("Creating file "+computedModelFileName+"...");
				File model = new File(computedModelFileName);
				model.createNewFile();
				FileWriter fileWriter = new FileWriter(model);
				BufferedWriter modelBufferedWriter = new BufferedWriter(fileWriter);
				
				while ((outputString = bufferedReader.readLine()) != null) {
					modelBufferedWriter.append(outputString);
					modelBufferedWriter.append(CRLF);
				}		
				
				modelBufferedWriter.close();
				long end=System.currentTimeMillis();
				m_modelComputationTime=end-start;			
			} catch (Exception e) {
				e.printStackTrace();
			}
			return new StableModel(computedModelFileName,false);
		}	
}
